/*= Instrument bp2304 =====================================================*/
/* LabWindows 2.3 Instrument Driver                                        */
/* Original Release: bp2304.c   V1.0    by MW            Date: 08.06.95    */
/*          Release: bp2304.c   V1.1    by MW            Date: 17.07.96    */
/*			Release: bp2304.c   V1.2	by MT			 Date: 01.08.2001  */
/*			Release: bp2304.c   V1.3	by MT			 Date: 03.02.2003  */
/*					   		(CloseCom added)							   */
/*					 													   */
/* By: M.Westermann                                                        */
/* Originally written in C                                                 */
/*=========================================================================*/

#include "bp2304.h"
#include <userint.h>
#include <rs232.h>
#include <formatio.h>
#include <utility.h>
#include <gpib.h>


/*= STATIC VARIABLES ======================================================*/

static int bp2304_err;				  						
static int interface;          				/* 0 = GPIB; 1 = Serial*/ 		
static int setup_first ;                  /* remark the first initial*/
static double std_timeout;                /* remark the user timeout*/
static int max_bcc_repeat;                /* Max-Bcc-Repeat */
static int bcc_repeat;                    /* Bcc-Repeat counter*/
static int bcc_state;                     /* Blockcheck state ON or OFF*/
static int instr_cnt;


/*= INSTRUMENT-DEPENDENT COMMAND ARRAYS ===================================*/
/* INSERT COMMAND ARRAY DECLARATIONS HERE. SEE THE BOTTOM OF THIS FILE FOR */
/* FURTHER EXPLANATION.                                                    */
/*=========================================================================*/





static int bd[bp2304_MAX_INSTR + 1];      /*Gpib Board*/	


static int address[bp2304_MAX_INSTR + 1];  /*GPIB-Adress*/
static char cmd[bp2304_MAX_CMD+1];
static char buf[bp2304_MAX_CMD+1];
static char out_buf[bp2304_MAX_CMD+1];       /* buffer for serial send */
static char in_buf[bp2304_MAX_CMD+1];        /* buffer for serial receive */
static char swap_buf[bp2304_MAX_CMD+1];      /* global serial buffer */
static int port_addr[4] ;                    /* RS232-Port Array*/
static int int_level[4] ;                    /* RS232-Interrupt-Level Array*/
static int dev_port[bp2304_MAX_INSTR];       /* Device Port*/
static int dev_group_addr[bp2304_MAX_INSTR]; /* Device Group serial Adress*/
static int dev_user_addr[bp2304_MAX_INSTR];  /* Device User serial Adress */

static char* sense_frequenz[3];
static char* sense_range[10];
static char* sense_resolution[3];
static char* source_shape[3];
static char* source_level[3];
static char* sense_loading[2];
static char* input_ground[2];
static char* system_keyb[3];

/*= UTILITY ROUTINES ======================================================*/

int _VI_FUNC bp2304_spoll(int instrID,unsigned char* srq); 
int _VI_FUNC bp2304_invalid_integer_range (int, int, int, int);
int _VI_FUNC bp2304_get_exe_error(int );                   
int _VI_FUNC bp2304_invalid_real_range (double, double, double, int);
int _VI_FUNC bp2304_read_std_event_reg(int,int* );         
int _VI_FUNC bp2304_open_instr_gpib(int );
int _VI_FUNC bp2304_read_oper_event_reg(int ,int*) ;       
int _VI_FUNC bp2304_device_closed_gpib (int);
int _VI_FUNC bp2304_close_instr_gpib (int);
int _VI_FUNC bp2304_read_msg_gpib (int,char*,int);
int _VI_FUNC bp2304_write_msg_gpib (int,char *, int);
void bp2304_setup_arrays (void);
int _VI_FUNC bp2304_get_status(int);
int _VI_FUNC bp2304_calculate_bcc(char* );
int _VI_FUNC bp2304_send_x3_28(int,char*,char*);
int _VI_FUNC bp2304_send_x3_28_message(int,char*,char*);
int _VI_FUNC bp2304_receive_x3_28(int,char*,char*,char*);
int _VI_FUNC bp2304_receive_and_ack(int,char*);
int _VI_FUNC bp2304_close_instr_serial (int);
int _VI_FUNC bp2304_device_closed_serial (int);
int _VI_FUNC bp2304_open_instr_serial (int,long,int,int);

/*=========================================================================*/
/* This function opens the instrument, queries the instrument for its ID,  */
/* and initializes the instrument to a known state.                        */
/*=========================================================================*/
int _VI_FUNC bp2304_init_gpib(int addr,int time_out,int reset,char *model,int* instrID)
{
    int ID;
    bp2304_err = 0;
    interface = bp2304_GPIB;            /* GPIB */

    if (bp2304_invalid_integer_range (addr, 0, 30,  -1) != 0)
    return bp2304_err;
    if (bp2304_invalid_integer_range (reset, 0, 1,  -2) != 0)
    return bp2304_err;

    ID = bp2304_open_instr_gpib (addr);
    if (ID <= 0)
      return bp2304_err;

    ibtmo (bd[ID], time_out);

    if (reset == 1)
    {
     if ((ibclr (bd[ID]) & (int)0x8000 != 0) != 0)
     {
      bp2304_close_instr_gpib (ID);
      bp2304_err = 224;
      return bp2304_err;
     }

     if (bp2304_send_scpi (ID,"*RST") != 0)
     {
      bp2304_close_instr_gpib (ID);
      return bp2304_err;
     }
     if (bp2304_send_scpi (ID,"*CLS") != 0)
     {
      bp2304_close_instr_gpib (ID);
      return bp2304_err;
     }
     
    }

    if (bp2304_receive_scpi(ID,"*IDN?",model) != 0)
    {
     bp2304_close_instr_gpib (ID);
     return bp2304_err;
    }
    /*  terminate read on LF  */
    if (ibeos (bd[ID], 0x40a) < 0)
    {
     bp2304_err = 237;
     return bp2304_err;
    }
     Fmt(buf,"STAT:OPER:ENAB %i",bp2304_EOC);/* EOC als Maske Freigeben*/
     if (bp2304_send_scpi (ID,buf) != 0)
     {
      bp2304_close_instr_gpib (ID);
      return bp2304_err;
     }
     Fmt(buf,"*ESE %i",bp2304_DEVICE_DEP_ERROR);/* Maske Freigeben*/
     if (bp2304_send_scpi (ID,buf) != 0)
     {
      bp2304_close_instr_gpib (ID);
      return bp2304_err;
     }
     /* Dev.Dep.Error + SRQ Register auch freigeben*/
     if (bp2304_send_scpi (ID,"*SRE 160") != 0)
     {
      bp2304_close_instr_gpib (ID);
      return bp2304_err;
     }

    *instrID = ID;

    bp2304_setup_arrays ();
    return(0);
}
/*=========================================================================*/
/* This function opens a com port for the instrument module                */
/*=========================================================================*/
int _VI_FUNC bp2304_init_serial(int port,long baud_rate,double timeout,int set_bcc_state,
           int g_addr,int u_addr,char* model,int reset,int *instrID)
{
    int ID;
    interface = bp2304_SERIAL;   /* Serial */

    bp2304_err = 0;
    bp2304_setup_arrays ();

    if (bp2304_invalid_integer_range (port, 1,4, -1) != 0)
     return bp2304_err;
    if (bp2304_invalid_integer_range (baud_rate, 1200, 19200, -2) != 0)
     return bp2304_err;
    if (bp2304_invalid_real_range (timeout, 0.0F, 72000.0F, -3) != 0)
     return bp2304_err;
    if (bp2304_invalid_integer_range (set_bcc_state, 0,1, -4) != 0)
     return bp2304_err;
    if (bp2304_invalid_integer_range (g_addr, 0,15, -5) != 0)
     return bp2304_err;
    if (bp2304_invalid_integer_range (u_addr, 0,15, -6) != 0)
     return bp2304_err;
    if (bp2304_invalid_integer_range (reset, 0,1, -7) != 0)
     return bp2304_err;

    ID = bp2304_open_instr_serial(port,baud_rate,g_addr,u_addr);
    if(ID <= 0)
     return bp2304_err;

    SetComTime (port, timeout);
    std_timeout = timeout;

    if (rs232err != 0)
     return (rs232err);

    max_bcc_repeat = 5;       /* default*/
    bcc_state = set_bcc_state;

    if (reset == 1)
    {
     if (bp2304_send_scpi (ID, "*RST") != 0)
     {
      bp2304_close_instr_serial(ID);
      return bp2304_err;
     }
    }
    if (bp2304_receive_scpi(ID,"*IDN?",model) != 0)
    {
     bp2304_close_instr_serial (ID);
     return bp2304_err;
    }
    *instrID = ID;

    return(0);
}
/* ========================================================================= */
/*  This function puts the device into local mode.                              */
/* ========================================================================= */
int _VI_FUNC bp2304_local_gpib(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
    return bp2304_err;
    if (bp2304_device_closed_gpib (instrID) != 0)
    return bp2304_err;
    if ((ibloc (bd[instrID]) & (int)0x8000 != 0) != 0)
    {
       bp2304_err = 234;
       return bp2304_err;
    }
    return bp2304_err;
}
/*=========================================================================*/
/* This function init the measurement sequence                            */
/*=========================================================================*/
int _VI_FUNC bp2304_config_meas_sequence(int instrID,int frequenz,int timebase,
                int average,int pause)
{
    bp2304_err = 0;

    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (frequenz, 0, 2,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (timebase, 1, 9,  -3) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (average, 1, 255,  -4) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (pause, 1, 255,  -5) != 0)
      return bp2304_err;

    if(bp2304_send_scpi(instrID,sense_frequenz[frequenz]) != 0) /* Frequenz*/
     return bp2304_err;

    Fmt(buf,"%s<:SENSE:RES:APER %i",timebase);
    if(bp2304_send_scpi(instrID,buf) != 0) /* Timebase*/
     return bp2304_err;

    Fmt(buf,"%s<:SENSE:RES:AVER %i",average);
    if(bp2304_send_scpi(instrID,buf) != 0) /* Average*/
     return bp2304_err;


    Fmt(buf,"%s<:SENSE:RES:PAUS %i",pause);
    if(bp2304_send_scpi(instrID,buf) != 0) /* Measure pause*/
     return bp2304_err;

    return(0);
}
/*=========================================================================*/
/* This function initialize the measurement mode                           */
/*=========================================================================*/
int _VI_FUNC bp2304_config_meas_mode(int instrID,int meas_mode,int range,int resolution)
{
    bp2304_err = 0;

    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (meas_mode, 0, 1,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (range, 0, 9,  -3) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (resolution, 0, 2,  -4) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:INIT:CONT %i",meas_mode);
    if(bp2304_send_scpi(instrID,buf) != 0) /* Mode*/
     return bp2304_err;

    if(bp2304_send_scpi(instrID,sense_range[range]) != 0) /* Range*/
     return bp2304_err;

    if(bp2304_send_scpi(instrID,sense_resolution[resolution]) != 0) /* Resolution*/
     return bp2304_err;

    return(0);
}
/*=========================================================================*/
/* This function initialize the measurement current                        */
/*=========================================================================*/
int _VI_FUNC bp2304_config_meas_current(int instrID,int shape,int level,int loading)
{
    bp2304_err = 0;

    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if (bp2304_invalid_integer_range (shape, 0, 2,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (level, 0, 2,  -3) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (loading, 0, 1,  -4) != 0)
     return bp2304_err;

    if(bp2304_send_scpi(instrID,source_shape[shape]) != 0) /* Shape*/
     return bp2304_err;

                        /* NO individual current*/
    if(bp2304_send_scpi(instrID,":SOUR:CURR:REF:STAT OFF") != 0)
     return bp2304_err;

    if(bp2304_send_scpi(instrID,source_level[level]) != 0)/* Level*/
     return bp2304_err;

    if(bp2304_send_scpi(instrID,sense_loading[loading]) != 0)/* Level*/
     return bp2304_err;

    return(0);
}
/*=========================================================================*/
/* This function initialize the measurement ground                         */
/*=========================================================================*/
int _VI_FUNC bp2304_config_meas_grounding(int instrID,int ground)
{
    bp2304_err = 0;

    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (ground, 0, 1,  -2) != 0)
      return bp2304_err;

    if(bp2304_send_scpi(instrID,input_ground[ground]) != 0)/* Ground*/
     return bp2304_err;


    return(0);
}
/*=========================================================================*/
/* This function start the measurnment                                     */
/*=========================================================================*/
int _VI_FUNC bp2304_meas_start(int instrID)
{

    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    
    if(bp2304_send_scpi(instrID,":INIT") != 0)
     return bp2304_err;

//    if(interface == bp2304_GPIB)  /* GPIB ?? */
//  {
    
//  }
    
    

    return (0);

}


int _VI_FUNC bp2304_spoll(int instrID,unsigned char* srq)
{
 
    ibrsp(bd[instrID], srq);
    if( ibsta & ERR )
    {
       bp2304_err = 304;
       return bp2304_err;
    }
    return 0;
}
/*=========================================================================*/
/* This function read the measurnment value with unit                      */
/*=========================================================================*/
int _VI_FUNC bp2304_meas_read(int instrID,double* value ,char* unit)
{
    int device_status,meas_status,ret;
    int done ,mask;
    unsigned char srq ;


    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    meas_status = 0;
    while(!(meas_status & bp2304_EOC))
    {
     /* IM GPIB-Betrieb darf der Messwert nicht direkt abgefragt werden,
        => Abfrage ber SRQ. Ansonsten wird der 2304 zugefahren*/

     if(interface == bp2304_GPIB)  /* GPIB ?? */
     {
      done = 0;
      mask = 0x4800;
      srq = 0;
      while(done != 1)
      {
       if (ibwait (bd[instrID], mask) <= 0)
       {
        bp2304_err = iberr;
        return bp2304_err;
       }

       if (ibrsp(bd[instrID], &srq) <= 0)
       {
         bp2304_err = iberr;
         return bp2304_err;
       }
       /*FmtOut("\n srq = %i", srq);*/
       switch(srq)
       {
        case 192:   /* RQS + Operation Status Register*/
         done = 1;
         ret = bp2304_read_oper_event_reg(instrID,&meas_status);
         if(ret != 0)
          return (ret);
         break;
        case 96:   /* RQS+ Standard Event Register*/
         done = 1;
         ret = bp2304_read_std_event_reg(instrID,&device_status);
         if(ret != 0)
          return (ret);

         if(device_status & bp2304_DEVICE_DEP_ERROR)
         {
          /*----------------- Messung stoppen wegen Fehler------------- */
          ret = bp2304_meas_stop(instrID);
          if(ret != 0)
           return (ret);

          ret = bp2304_system_error(instrID);
         if((ret >= 10) && (ret <= 60))
          ret += 300;
         return (ret);
        }
        break;
       case 0:   /* no Value in Srq - Byte*/
        /*FmtOut("\nsrq %i meas_status %i",srq,meas_status);*/
         if(!(meas_status & bp2304_MEASURING))    /* no Meas Flag*/
          return 302;
         break;
       }
      }
     }
     else
     {
      ret = bp2304_read_std_event_reg(instrID,&device_status);
      if(ret != 0)
       return (ret);

      if(device_status & bp2304_DEVICE_DEP_ERROR)
      {
      /*----------------- Messung stoppen wegen Fehler------------- */
      ret = bp2304_meas_stop(instrID);
      if(ret != 0)
       return (ret);

      /*------- Dummy Abfrage des DEVICE_DEP_ERROR wegen Zeitprobleme ----*/
      ret = bp2304_read_std_event_reg(instrID,&device_status);
      if(ret != 0)
       return (ret);

      ret = bp2304_system_error(instrID);
      if((ret >= 10) && (ret <= 60))
       ret += 300;
      return (ret);
     }

      /*------ Das Operation Status Register den EOC des 2304 abfragen-------- */
      ret = bp2304_read_oper_event_reg(instrID,&meas_status);
      if(ret != 0)
       return (ret);

      if(!(meas_status & bp2304_EOC))    /* no EOC*/
      {
/*      FmtOut("\nret %i meas_status %i",ret,meas_status);*/
       if(!(meas_status & bp2304_MEASURING))    /* no Meas Flag*/
        return 302;
      }
     }
    }

   /*------------------ Den Messwert des 2304 abfragen--------------- */

    if(bp2304_receive_scpi(instrID,":FETCH?",buf) != 0)
     return bp2304_err;

    Scan(buf,"%s>%f%s",value,unit);

    return (0);
}

/*=========================================================================*/
/* This function stop the measurnment                                     */
/*=========================================================================*/
int _VI_FUNC bp2304_meas_stop(int instrID)
{
    unsigned char srq ;
    int meas_status;
    
    bp2304_err = 0;
    
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if(bp2304_send_scpi(instrID,":ABORT") != 0)
     return bp2304_err;

    if(interface == bp2304_GPIB)  /* GPIB ?? */
    {
     /*-------- die SRQ-Leitung wieder freigegeben -------- */
     //if (ibrsp(bd[instrID], &srq) <= 0)
     //{
     //  bp2304_err = iberr;
     //  return bp2304_err;
     //}
     
	}
    return (0);
}

/*=========================================================================*/
/* This function set the coefficient value                                 */
/*=========================================================================*/
int _VI_FUNC bp2304_corr_set_tk(int instrID,int tk_number,int value)

{
    bp2304_err = 0;

    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (tk_number, 1, 10,  -2) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:SENS:CORR:TCOM:TCO%i %i",tk_number,value);
    if(bp2304_send_scpi(instrID,buf) != 0)/* set TK*/
     return bp2304_err;

    return(0);
}
/*=========================================================================*/
/* This function set the coefficient value                                 */
/*=========================================================================*/
int _VI_FUNC bp2304_corr_state(int instrID,int state,int flag,double value)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state, 0, 1,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (flag, 0, 1,  -3) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:SENS:CORR:TCOM:AUTO %i",flag);
    if(bp2304_send_scpi(instrID,buf) != 0)/* set Auto ON/OFF*/
     return bp2304_err;

    if(flag == 0)    /* Manual */
    {
     if (bp2304_invalid_real_range (value, -70, 999.99F,  -4) != 0)
      return bp2304_err;

     Fmt(buf,"%s<:SENS:CORR:TCOM:MAN %f",value);
     if(bp2304_send_scpi(instrID,buf) != 0)/* set TK Manual*/
      return bp2304_err;
    }

    Fmt(buf,"%s<:SENS:CORR %i",state);
    if(bp2304_send_scpi(instrID,buf) != 0)/* set State ON/OFF*/
     return bp2304_err;

    return(0);
}
/*=========================================================================*/
/* This function get the temperature value                                 */
/*=========================================================================*/
int _VI_FUNC bp2304_get_temp(int instrID,double* value)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if(bp2304_receive_scpi(instrID,":SENS:CORR:TCOM:MAN?",buf) != 0)
     return bp2304_err;

    Scan(buf,"%s>%fCEL",value);

    return(0);
}
/*=========================================================================*/
/* This function set the comparator limit                                  */
/*=========================================================================*/
int _VI_FUNC bp2304_comp_limit(int instrID,int state,int menu_bar,double l_limit,
            double h_limit,double nominal,int unit)
{
    int ret,err;

    err = 0;
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state,0, 1,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (menu_bar,0, 1,  -3) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (unit,0, 1,  -4) != 0)
      return bp2304_err;


    Fmt(buf,"%s<:SORT:STATE %i",state);         /* Sort ON/OFF*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    if(state == 1)   /* Onlay by State == ON*/
    {
     Fmt(buf,"%s<:DISP:MENU %i",menu_bar);         /* Display bar ON/OFF*/
     if(bp2304_send_scpi(instrID,buf) != 0)
      return bp2304_err;

     Fmt(buf,"%s<:SORT:COMP:NOM %f",nominal);
     if(bp2304_send_scpi(instrID,buf) != 0)
      return bp2304_err;

     if(unit == 1)    /* Percent*/
     {
      Fmt(buf,"%s<:SORT:COMP:LLIMIT %fPCT",l_limit);
      if(bp2304_send_scpi(instrID,buf) != 0)
       return bp2304_err;

      Fmt(buf,"%s<:SORT:COMP:HLIMIT %fPCT",h_limit);
      if(bp2304_send_scpi(instrID,buf) != 0)
       return bp2304_err;
     }
     else
     {
      Fmt(buf,"%s<:SORT:COMP:LLIMIT %f",l_limit);
      if(bp2304_send_scpi(instrID,buf) != 0)
       return bp2304_err;

      Fmt(buf,"%s<:SORT:COMP:HLIMIT %f",h_limit);
      if(bp2304_send_scpi(instrID,buf) != 0)
       return bp2304_err;
     }

     ret = bp2304_send_scpi(instrID,":SORT:COMP:ACKN");
     if(ret != 0)           /* Witch Error */
     {
      err = bp2304_get_exe_error(instrID);
       return (err);
     }
    }
    return(0);
}
/*=========================================================================*/
/* This function set the comparator reset                                  */
/*=========================================================================*/
int _VI_FUNC bp2304_comp_reset(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if(bp2304_send_scpi(instrID,":SORT:COMP:DATA:RESET") != 0)
     return bp2304_err;

    return(0);
}
/*=========================================================================*/
/* This function get the comparator statistic data                         */
/*=========================================================================*/
int _VI_FUNC bp2304_get_comp_data(int instrID,int value[])
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if(bp2304_receive_scpi(instrID,":SORT:COMP:DATA?",buf) != 0)
     return bp2304_err;

    Scan(buf,"%s>%4i[x]",value);

    return(0);
}
/*=========================================================================*/
/* This function set the class.comparator limit                                  */
/*=========================================================================*/
int _VI_FUNC bp2304_class_comp_limit(int instrID,int state,int menu_bar,
                    double limit[],double nominal,int unit)
{
    int err,i;

    err = 0;
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state,0, 1,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (menu_bar,0, 1,  -3) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (unit,0, 1,  -4) != 0)
      return bp2304_err;


    Fmt(buf,"%s<:SORT:STATE %i",state);         /* Sort ON/OFF*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    if(state == 1)         /* only by state = on*/
    {
     Fmt(buf,"%s<:DISP:MENU %i",menu_bar);         /* Display bar ON/OFF*/
     if(bp2304_send_scpi(instrID,buf) != 0)
      return bp2304_err;

     Fmt(buf,"%s<:SORT:BINN:NOM %f",nominal);
     if(bp2304_send_scpi(instrID,buf) != 0)
      return bp2304_err;

     if(unit == 1)    /* Percent*/
     {
      for(i = 0;i < bp2304_MAX_BINN_LIMIT;i++)
      {
       Fmt(buf,"%s<:SORT:BINN:LIMIT%d %fPCT",i+1,limit[i]);
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
      }
     }
     else
     {
      for(i = 0;i < bp2304_MAX_BINN_LIMIT;i++)
      {
       Fmt(buf,"%s<:SORT:BINN:LIMIT%d %f",i+1,limit[i]);
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
      }
     }

     if(bp2304_send_scpi(instrID,":SORT:BINN:ACKN") != 0)
      return bp2304_err;

     err = bp2304_get_exe_error(instrID);
    }
    return(err);
}
/*=========================================================================*/
/* This function set the class comparator reset                            */
/*=========================================================================*/
int _VI_FUNC bp2304_class_comp_reset(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if(bp2304_send_scpi(instrID,":SORT:BINN:DATA:RESET") != 0)
     return bp2304_err;

    return(0);
}
/*=========================================================================*/
/* This function get the class comparator statistic data                   */
/*=========================================================================*/
int _VI_FUNC bp2304_get_class_comp_data(int instrID,int value[])
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if(bp2304_receive_scpi(instrID,":SORT:BINN:DATA?",buf) != 0)
     return bp2304_err;

    Scan(buf,"%s>%10i[x]",value);

    return(0);
}
/*=========================================================================*/
/* This function initialize cooling curve state                            */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_state(int instrID,int state)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state,0, 1,  -2) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:CCUR:STATE %i",state);     /* Cooling Curve Menu ON/OFF*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function initialize meas cooling curve                             */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_meas_init(int instrID,int clear,double cool_delay)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if (bp2304_invalid_integer_range (clear, 0, 1,  -2) != 0)
      return bp2304_err;

    if(clear == 1)
    {
     if(bp2304_send_scpi(instrID,":CCUR:CLE") != 0)
      return bp2304_err;
    }

    Fmt(buf,"%s<:CCUR:DEL %f",cool_delay);     /* Cooling Curve delay*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}

/*=========================================================================*/
/* This function manual cooling curve Rk                                   */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_man_Rk(int instrID, double rk)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:CCUR:CRES %f",rk);     /* Cooling Curve Rk*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function meas cooling curve Rk start                               */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_meas_Rk_start(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:CCUR:CRES:INIT\n");     /* Cooling Curve Rk Start*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function meas cooling curve Rk stop                                */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_meas_Rk_stop(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:CCUR:CRES:ABOR\n");     /* Cooling Curve Rk Stop*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function read cooling curve Rk                                     */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_read_RK(int instrID,double* value,char* unit)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

             /* Read Cooling Curve Rk */
    if(bp2304_receive_scpi(instrID,"CCUR:CRES?",buf) != 0)
     return bp2304_err;

    Scan(buf,"%s>%f%s",value,unit);

    return (0);
}

/*=========================================================================*/
/* This function measure cooling curve T1                                  */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_meas_t1(int instrID, int meas_mode,double value)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if (bp2304_invalid_integer_range (meas_mode, 0, 1,  -2) != 0)
      return bp2304_err;

    if(meas_mode == 1)
    {
                           /* Cooling Curve T1 auto*/
     if(bp2304_send_scpi(instrID,"CCUR:BTEM:INIT") != 0)
      return bp2304_err;
    }
    else
    {
     Fmt(buf,"%s<:CCUR:BTEM %f",value);     /* Cooling Curve T1 maunal*/
     if(bp2304_send_scpi(instrID,buf) != 0)
      return bp2304_err;
    }
    return (0);
}
/*=========================================================================*/
/* This function read cooling curve T1                                     */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_read_t1(int instrID,double* value)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
                                     /* Read Cooling Curve T1 */
    if(bp2304_receive_scpi(instrID,"CCUR:BTEM?",buf) != 0)
     return bp2304_err;

    Scan(buf,"%s>%fCEL",value);

    return (0);
}

/*=========================================================================*/
/* This function measure cooling curve T2                                  */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_meas_t2(int instrID,int meas_mode,double value)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    if (bp2304_invalid_integer_range (meas_mode, 0, 1,  -3) != 0)
      return bp2304_err;

    if(meas_mode == 1)
    {
                           /* Cooling Curve T2 auto*/
     if(bp2304_send_scpi(instrID,"CCUR:ETEM:INIT") != 0)
      return bp2304_err;
    }
    else
    {
     Fmt(buf,"%s<:CCUR:ETEM %f",value);     /* Cooling Curve T2 maunal*/
     if(bp2304_send_scpi(instrID,buf) != 0)
      return bp2304_err;
    }
    return (0);
}
/*=========================================================================*/
/* This function read cooling curve T2                                     */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_read_t2(int instrID,double* value)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
                                     /* Read Cooling Curve T1 */
    if(bp2304_receive_scpi(instrID,"CCUR:ETEM?",buf) != 0)
     return bp2304_err;

    Scan(buf,"%s>%fCEL",value);

    return (0);
}

/*=========================================================================*/
/* This function measure cooling curve Charge OFF                          */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_removal_load(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:CCUR:CHAR OFF");     /* Cooling Curve Removal load*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function measure cooling curve Start                               */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_meas_start(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;


    Fmt(buf,"%s<:CCUR:INIT");     /* Cooling Curve measure start*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function measure cooling curve Stop                               */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_meas_stop(int instrID)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:CCUR:ABOR\n");     /* Cooling Curve measure stop*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function measure cooling curve Data                               */
/*=========================================================================*/
int _VI_FUNC bp2304_cool_read_meas(int instrID,double ccurve_time[],
                            double ccurve_value[],char * unit ,int * number)
{
    char dummy[10],msg[5];
    int i,l,ret, c;


    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;

    /* Das Auslesen von Daten aus dem Speicher mit RS232 und Gpib unter-
       scheidet sich sehr. Bei Gpib wrden die Daten auf einmal kommen, aber
       man msste einen groen Speicherplatz dafr vorsehen, deshalb wird
       Byteweise eingelesen und nach dem Komma die Werte in die Array kopiert.
       Bei RS232 mssen die Daten Blockweise bei EOT geholt werden.*/

    if(interface == bp2304_GPIB)  /* GPIB ?? */
    {
     if(bp2304_send_scpi(instrID,"CCUR:DATA?") != 0)
      return bp2304_err;

     msg[0] = 0x00;
     l = 0;
     while((msg[0] != 0x0A)) /* ende bei LF*/
     {
      msg[0] = 0x00;
      i = 0;
      while((msg[0] != ',') && (msg[0] != 0x0A)) /* einlesen bis komma oder LF */		 
      {
       c = ibrd (bd[instrID], msg, 1L);
       if (c <= 0) /* nur ein Byte einlesen*/
       {
          if( l ==  0)
           bp2304_err = 303;
          else
           bp2304_err = 231;

          return bp2304_err;
       }
       else if (ibcntl == 0)   // MT		  // Es wurde auch eingelesen, wenn keine Daten vorhanden waren
       {					   // MT		  // dadurch blieb die Schnittstelle hngen !
        bp2304_err = 303;	   // MT  		  // 2001-01-24 MT
        return bp2304_err;	   // MT
       }					   // MT
       buf[i] = msg[0];   /* einzlnes Byte speichern */
/*       FmtOut("\n %i msg [0] %c buf '%s' \n",i,msg[0],buf);*/
       i++;
      }
     /* FmtOut("\nbuf '%s' l = %i\n",buf,l);*/  /* Werte ein Arrays speichern */
      Scan(buf,"%s>%f%s %f%s",&ccurve_time[l],dummy,&ccurve_value[l],unit);
      l++;
     }
     *number = l;   /* gesamt Anzahl bergeben */

    }
    else
    {
     if(bp2304_send_scpi(instrID,"CCUR:DATA?") != 0) /* Anforderung */
      return bp2304_err;

     FlushInQ(dev_port[instrID]);           /* Clear Port */
     ret = bp2304_send_x3_28_message(instrID,"po","");   /* Werte abholen */
     if(ret != 0)
      return (ret);     /* ERROR*/

     ret = 0;
     i = 0;
     while(ret != 301)   /* EOT =  no more data => bis EOT einlesen*/
     {
      ret = bp2304_receive_and_ack(instrID,buf);  /* Werte anholen und ACK*/
      if(ret == 0)
      {
      /* FmtOut("\nData '%s' %i %i\n",buf,ret,i);*/
       Scan(buf,"%s>%f%s %f%s",&ccurve_time[i],dummy,&ccurve_value[i],unit);
       i++;
      }
     }
     *number = i; /* Gesamtanzahl bergeben*/
     if(( i == 0) && (ret == 301))
      return 303;
    }
    return (0);
}
/*=========================================================================*/
/* This function set the display mode                                      */
/*=========================================================================*/
int _VI_FUNC bp2304_disp_mode(int instrID,int state,int disp_mode,double nominal,
                     double length,double cross_section,double density,
                     double mass, double measlength)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state, 0, 1,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (disp_mode, 0, 8,  -3) != 0)
      return bp2304_err;



    Fmt(buf,"%s<:CALC:STATE OFF");         /* State OFF */
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    if(state == 1)    /* ON */
    {
     switch(disp_mode)
     {
      case 0:         /* Ohm/m */
       if(bp2304_send_scpi(instrID,":CALC:MATH R/LENGTH0[M]") != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH0,%f\n",length);         /* Length */
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break;
      
      case 1:         /* Ohm/km */
       if(bp2304_send_scpi(instrID,":CALC:MATH R/LENGTH0[KM]") != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH0,%f\n",length);         /* Length*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break ;
       
      case 2:         /* Ohm/10ft */
       if(bp2304_send_scpi(instrID,":CALC:MATH R/LENGTH0[10FT]") != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH0,%f\n",length);         /* Length*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break ;
       
      case 3:         /* Ohm/kft */
       if(bp2304_send_scpi(instrID,":CALC:MATH R/LENGTH0[KFT]") != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH0,%f\n",length);         /* Length*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break ;

      case 4:     /* Delta% */
       Fmt(buf,"%s<:CALC:MATH DELTA%%");
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA NOMINAL,%f\n",nominal);   /* Nominal Value*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break ;
      
      case 5:             /* RHO1 */
       Fmt(buf,"%s<:CALC:MATH RHO1");
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH1,%f",length);         /* Length*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA AREA,%f",cross_section);      /*cross section*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break ;
       
      case 6:    /*RHO2*/
       Fmt(buf,"%s<:CALC:MATH RHO2");
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA MASS,%f",mass);      /*mass*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH2,%f",length);         /* Length*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH3,%f",measlength);         /* Measlength*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA DENSITY,%f",density);      /*Density*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break ;
      
      case 7:		// Gamma 1
       Fmt(buf,"%s<:CALC:MATH GAMMA1");
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH1,%f",length);         /* Length*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA AREA,%f",cross_section);      /*cross section*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;
       break ;
      
      case 8:	  	// Gamma 2
       Fmt(buf,"%s<:CALC:MATH GAMMA2");
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;


       Fmt(buf,"%s<:TRACE:DATA MASS,%f",mass);      /*mass*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH2,%f",length);         /* Length*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA LENGTH3,%f",measlength);         /* Measlength*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       Fmt(buf,"%s<:TRACE:DATA DENSITY,%f",density);      /*Density*/
       if(bp2304_send_scpi(instrID,buf) != 0)
        return bp2304_err;

       break ;
     }
     Fmt(buf,"%s<:CALC:STATE ON\n");         /* State ON*/
     if(bp2304_send_scpi(instrID,buf) != 0)
      return bp2304_err;
    }

    return(0);
}
/*=========================================================================*/
/* This function set the display enable                                    */
/*=========================================================================*/
int _VI_FUNC bp2304_disp_enable(int instrID,int state)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state, 0, 1,  -1) != 0)
      return bp2304_err;


    Fmt(buf,"%s<:DISP:ENAB %i\n",state);         /* Display Enable ON/OFF*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function set the display contrast                                   */
/*=========================================================================*/
int _VI_FUNC bp2304_disp_contrast(int instrID,double contrast)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_real_range (contrast, 0.0, 1.0,  -2) != 0)
      return bp2304_err;


    Fmt(buf,"%s<:DISP:CONT %f",contrast);         /* Display contrast*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function set the output pin 3+21                                   */
/*=========================================================================*/
int _VI_FUNC bp2304_system_output(int instrID,int state_pin3,int state_pin21)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state_pin3, 0, 1,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state_pin21, 0, 1,  -3) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:OUTP:TTLTRG0:LEVEL %i",state_pin21);  /* Pin 21*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;


    Fmt(buf,"%s<:OUTP:TTLTRG1:LEVEL %i",state_pin3);  /* Pin 3*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function set the keyboard control                                  */
/*=========================================================================*/
int _VI_FUNC bp2304_system_keyboard(int instrID,int lock,int access)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (access, 0, 2,  -2) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (lock, 0, 1,  -3) != 0)
      return bp2304_err;

    if(bp2304_send_scpi(instrID,system_keyb[access]) != 0)
     return bp2304_err;

    Fmt(buf,"%s<:SYST:KLOC %i\n",lock);  /* Locked/Unlocked*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}

/*=========================================================================*/
/* This function set the buzzer         ON/OFF                             */
/*=========================================================================*/
int _VI_FUNC bp2304_system_buzzer(int instrID,int state)
{
    bp2304_err = 0;
    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
      return bp2304_err;
    if (bp2304_invalid_integer_range (state, 0, 1,  -2) != 0)
      return bp2304_err;

    Fmt(buf,"%s<:SYSTEM:BEEPER %i",state);  /* Buzzer ON/OFF*/
    if(bp2304_send_scpi(instrID,buf) != 0)
     return bp2304_err;

    return (0);
}
/*=========================================================================*/
/* This function get execution system error                                */
/*=========================================================================*/
int _VI_FUNC bp2304_get_exe_error(int instrID)
{
   int ret;
   int device_status;

   bp2304_err = 0;
   if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
     return bp2304_err;


   device_status = 0;

   ret = bp2304_read_std_event_reg(instrID,&device_status) ;
   if(ret != 0)
    return (ret);

   if(device_status & bp2304_EXECUATION_ERROR)
   {
     ret = bp2304_system_error(instrID);
     return (ret);
   }
   return (0);
}
/*=========================================================================*/
/* This function read the Standrad event register                          */
/*=========================================================================*/
int _VI_FUNC bp2304_read_std_event_reg(int instrID,int* status)
{
   bp2304_err = 0;
   if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
     return bp2304_err;

   if(bp2304_receive_scpi(instrID,"*ESR?",buf) != 0)
    return (bp2304_err);

   Scan(buf,"%s>%i",status);

   return (0);
}
/*=========================================================================*/
/* This function read the operation event register                         */
/*=========================================================================*/
int _VI_FUNC bp2304_read_oper_event_reg(int instrID,int* status)
{
   bp2304_err = 0;
   if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
     return bp2304_err;

   if(bp2304_receive_scpi(instrID,":STAT:OPER:EVEN?",buf) != 0)
    return (bp2304_err);

   Scan(buf,"%s>%i",status);

   return (0);
}
/*=========================================================================*/
/* This function get system error                                          */
/*=========================================================================*/
int _VI_FUNC bp2304_system_error(int instrID)
{
   int err ,ret_error,found_error;

   bp2304_err = 0;
   if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
     return bp2304_err;

   err = -1;
   ret_error = 0;
   found_error = 0;

   while(err != 0) /* Einlesen bis kein Fehler kommt */
   {
    if(bp2304_receive_scpi(instrID,"SYST:ERR?",buf) != 0)
     return (bp2304_err);

    Scan(buf,"%s>%i",&err);
	if (buf[0] == 0)				// MT
		err = 0;					// MT

/*    FmtOut("\nSys %s",buf);*/
    if((err != 0) && (found_error == 0))
    {
     found_error = 1;    /* Save only the first Error */
     ret_error = err;
    }
   }
   return (ret_error);
}
/*=========================================================================*/
/* This function SendSCPI                                                  */
/*=========================================================================*/
int _VI_FUNC bp2304_send_scpi(int instrID,char* send_buffer)
{
   bp2304_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);


   if(interface == bp2304_GPIB)  /* GPIB ?? */
   {
    bp2304_err = bp2304_write_msg_gpib(instrID,cmd,NumFmtdBytes ());

    if(bp2304_err != 0)
     return (bp2304_err);
   }
   else    /* Serial */
   {
    bp2304_err = bp2304_send_x3_28(instrID,"sr",cmd);
    if(bp2304_err != 0)
     return (bp2304_err);
   }

   return(0);
}
/*=========================================================================*/
/* This function receive SCPI                                              */
/*=========================================================================*/
int _VI_FUNC bp2304_receive_scpi(int instrID,char* send_buffer,char* receive_buffer)
{
   bp2304_err = 0;

   Fmt(cmd,"%s<%s\n",send_buffer);



   if(interface == bp2304_GPIB)  /* GPIB ?? */
   {
    bp2304_err = bp2304_write_msg_gpib(instrID,cmd,NumFmtdBytes ());
    if(bp2304_err != 0)
     return (bp2304_err);

    bp2304_err = bp2304_read_msg_gpib(instrID,receive_buffer,bp2304_MAX_CMD);
    if(bp2304_err != 0)
     return (bp2304_err);
   }
   else
   {
    bp2304_err = bp2304_send_x3_28(instrID,"sr",cmd);
    if(bp2304_err != 0)
     return (bp2304_err);

    bp2304_err = bp2304_receive_x3_28(instrID,"po","",receive_buffer);
    if(bp2304_err != 0)
     return (bp2304_err);
   }


   return(0);
}
/* ========================================================================= */
/*  This function closes the instrument.                                     */
/* ========================================================================= */
int _VI_FUNC bp2304_close (int instrID)
{
    bp2304_err = 0;

    if (bp2304_invalid_integer_range (instrID, 1, bp2304_MAX_INSTR,  -1) != 0)
     return (bp2304_err);

    if(interface == bp2304_GPIB)  /* GPIB ?? */
    {
     if (bp2304_device_closed_gpib (instrID) != 0)
      return (bp2304_err);
     bp2304_close_instr_gpib (instrID);
    }
    else
    {
     if (bp2304_device_closed_serial (instrID) != 0)
      return (bp2304_err);
     bp2304_close_instr_serial (instrID);
    }
    return (bp2304_err);
}
/* = GPIB UTILITY ROUTINES == ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the GPIB device table for the instrument.  If successful, the        */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp2304_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC bp2304_open_instr_gpib(int addr)
{
    int i;
    int instrID;

    instrID = 0;
    /*  Check to see if the instrument is already in the Instrument Table.  */
    for (i = 1; i <= bp2304_MAX_INSTR; i++)
    if (address[i] == addr)
    {
        instrID = i;
        i = bp2304_MAX_INSTR;
    }
    /*  If it is not in the instrument table, open an entry for the instrument.  */
    if (instrID <= 0)
    for (i = 1; i <= bp2304_MAX_INSTR; i++)
        if (!address[i])
        {
        instrID = i;
        i = bp2304_MAX_INSTR;
        }
    /*  If an entry could not be opened in the Instrument Table, return an error. */
    if (instrID <= 0)
    {
    bp2304_err = 220;
    return  -1;
    }
    /*   If the device has not been opened in the GPIB device table (bd[ID] = 0), */
    /*   then open it.                                                            */
    if (bd[instrID] <= 0)
    {
    if (instr_cnt <= 0)
        CloseInstrDevs ("bp2304");
    bd[instrID] = OpenDev ("", "bp2304");
    if (bd[instrID] <= 0)
    {
        bp2304_err = 220;
        return  -1;
    }
    }
    /*   Change the primary address of the device     */
    if (ibpad (bd[instrID], addr) < 0)
    {
    bp2304_err = 233;
    return  -1;
    }
    address[instrID] = addr;
    instr_cnt++;
    return instrID;
}

/* ========================================================================== */
/*  This function closes the instrument by removing it from the GPIB device   */
/*  table and setting the address and the bd to zero in the Instrument Table. */
/*  The return value is equal to the global error variable.                   */
/* ========================================================================== */
int _VI_FUNC bp2304_close_instr_gpib (int instrID)
{

    if (bd[instrID] != 0)  {
    CloseDev (bd[instrID]);
    bd[instrID] = 0;
    address[instrID] = 0;
    instr_cnt--;
    }
    else
    bp2304_err = 221;
    return bp2304_err;
}
/* ========================================================================= */
/*  This function checks to see if the module has been initialized.  If the  */
/*  device has not been opened, a 1 is returned, 0 otherwise.                */
/* ========================================================================= */
int _VI_FUNC bp2304_device_closed_gpib (int instrID)
{

    if (bd[instrID] <= 0)
    {
    bp2304_err = 232;
    return  -1;
    }
    return 0;
}

/* ========================================================================= */
/*  This function reads a buffer of data from the instrument. The return     */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC bp2304_read_msg_gpib(int instrID,char *msg,int cnt)
{
    if (ibrd (bd[instrID], msg, (long)cnt) <= 0)
        bp2304_err = 231;
    else
        bp2304_err = 0;

	if (ibcntl == 0)					// MT
	{
		msg[ibcntl] = 0x00;				// MT
		//bp2304_err = 231;				// MT
	}
	
	if (ibcntl >= 2)					// MT
		msg[ibcntl-2] = 0x00;			// MT

    /*FmtOut("\n%s",msg);*/

    return bp2304_err;
}

/* ========================================================================= */
/*  This function writes a buffer of data to the instrument. The return      */
/*  value is equal to the global error variable.                             */
/* ========================================================================= */
int _VI_FUNC bp2304_write_msg_gpib (int instrID, char *msg, int cnt)
{
/*    FmtOut("\n%s",msg);*/

	 ibwrt (bd[instrID], msg, (long)cnt);

     if (iberr != 0)				  
       bp2304_err = 230;
     else
       bp2304_err = 0;

  //  if (ibwrt (bd[instrID], msg, (long)cnt) <= 0)
  //  bp2304_err = 230;
  //  else
  //  bp2304_err = 0;
    
    return bp2304_err;
}
/* = SERIAL UTILITY ROUTINES ============================================== */
/* ========================================================================= */
/*  This function locates and initializes an entry in the Instrument Table   */
/*  and the device table for the instrument.  If successful, the            */
/*  instrument ID is returned, else a -1 is returned.  The size of the       */
/*  Instrument Table can be changed in the include file by altering the      */
/*  constant bp2304_MAX_INSTR.                                               */
/* ========================================================================= */
int _VI_FUNC bp2304_open_instr_serial (int port,long baud_rate,int g_addr,int u_addr)
{
    int i, instrID;

    instrID = 0;

    /*--Check to see if the instrument is already in the Instrument Table.-*/

    for (i=1; i< bp2304_MAX_INSTR; i++)
    {
/*    FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
     if((dev_group_addr[i] == g_addr) && (dev_user_addr[i] == u_addr))
     {
      instrID = i;
      i = bp2304_MAX_INSTR;
     }
    }

   /*------ If it is not in the instrument table, open an entry for the
        instrument. ---------------------------------------------------*/

    if (instrID <= 0)
    {
     for (i=1; i< bp2304_MAX_INSTR; i++)
     {
/*     FmtOut("\n g_adr %i u_adr %i i= %i",dev_group_addr[i],dev_user_addr[i],i);*/
      if( (dev_group_addr[i] < 0) && (dev_user_addr[i] < 0))
      {
       instrID = i;
       i = bp2304_MAX_INSTR;
      }
     }
    }

    /*----- If an entry could not be opened in the Instrument Table,
        return an error.-----------------------------------------------*/

    if (instrID <= 0)
    {
     bp2304_err = 220;
     return -1;
    }


    dev_port[instrID] = port;
    OpenComConfig (dev_port[instrID], "",baud_rate,0, 8, 1, 512, 512);

    if (rs232err != 0)
    {
     CloseCom (dev_port[instrID]);		// gendert am 03.02.2003 / MT

     dev_port[instrID] = 0;
     dev_group_addr[instrID] = -1;
     dev_user_addr[instrID] = -1;
     bp2304_err = rs232err;
     return (-1);
    }
    dev_group_addr[instrID] = g_addr;
    dev_user_addr[instrID] = u_addr;
    return instrID;
}
/*=========================================================================*/
/* Function: Close Instrument                                              */
/* Purpose:  This function closes the instrument by removing it from the   */
/*           device table and setting the address and the bd to zero       */
/*           in the Instrument Table.  The return value is equal to the    */
/*           global error variable.                                        */
/*=========================================================================*/
int _VI_FUNC bp2304_close_instr_serial (int instrID)
{
    if( (dev_group_addr[instrID] >= 0) && (dev_user_addr[instrID] >= 0))
    {
      CloseCom (dev_port[instrID]);		// gendert am 03.02.2003 / MT

      dev_port[instrID] = 0;
      dev_group_addr[instrID] = -1;
      dev_user_addr[instrID] = -1;
    }
    else
     bp2304_err = 221;

    return bp2304_err;
 }
/*=========================================================================*/
/* Function: Device Closed                                                 */
/* Purpose:  This function checks to see if the module has been            */
/*           initialized.  If the device has not been opened, a 1 is       */
/*           returned, 0 otherwise.                                        */
/*=========================================================================*/
int _VI_FUNC bp2304_device_closed_serial (int instrID)
{
     if( (dev_group_addr[instrID] < 0) && (dev_user_addr[instrID] < 0))
     {
    bp2304_err = 232;
    return -1;
     }
     return 0;
}

/*=========================================================================*/
/* This function send a comand and message and wait for ACK                */
/*=========================================================================*/
int _VI_FUNC bp2304_send_x3_28(int instrID,char* cmd,char* msg)
{
    int ret,r_byte;

    FlushInQ(dev_port[instrID]);           /* Clear Port */

    bcc_repeat = 0;
    for(;;)
    {
     ret = bp2304_send_x3_28_message(instrID,cmd,msg);
     if(ret != 0)
       return (ret);     /* ERROR*/

     r_byte = ComRdByte(dev_port[instrID]);
     if (rs232err != 0)
      return (231);

    /* FmtOut("\nsend ret %c %d\n",r_byte,bcc_repeat);*/
     Delay(0.01);
     switch(r_byte)
     {
      case ACK:
       return  (0);
       break;
      case NAK:
       if(bcc_repeat >= max_bcc_repeat)
    return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
       break;
      case EOT:
       return (301);
       break;
     }
    }

    return (0);
}

/*=========================================================================*/
/* This function send comand and message and receive a string              */
/*=========================================================================*/
int _VI_FUNC bp2304_receive_x3_28(int instrID,char* cmd,char* msg,char* receive_buf)
{
    int ret;

    FlushInQ(dev_port[instrID]);           /* Clear Port */
    ret = bp2304_send_x3_28_message(instrID,cmd,msg);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ret = bp2304_receive_and_ack(instrID,receive_buf);
    if(ret != 0)
      return (ret);     /* ERROR*/

    ComRdByte(dev_port[instrID]);   /* Read the EOT from Port*/

    return(0);
}
/*=========================================================================*/
/* This function receive a string                                          */
/*=========================================================================*/
int _VI_FUNC bp2304_receive_and_ack(int instrID,char* receive_buf)
{
    int r_byte,r_bcc,bcc,l,first_byte;

    bcc_repeat = 0;
    for(;;)
    {
     first_byte = ComRdByte(dev_port[instrID]);   /* Read the EOT or STX from Port*/
     if(first_byte == EOT)
      return (301);

     r_byte = ComRdTerm(dev_port[instrID],in_buf,bp2304_MAX_CMD-1,ETX);/* Read to ETX*/
     if (rs232err != 0)
      return (231);

     if(bcc_state == 1)
     {
      r_bcc = ComRdByte(dev_port[instrID]);   /* Read the BCC from Port*/
      if (rs232err != 0)
       return (231);
     }
     in_buf[r_byte] = ETX;            /* Calculate bcc with ETX */
     in_buf[r_byte+1] = NUL;          /* Terminate the String*/
     l = StringLength(in_buf);

    /* FmtOut("\nin %c %s",first_byte,in_buf);*/

     CopyString (swap_buf, 0,in_buf, 0, l);   /* calculate bcc without STX */
     if(bcc_state == 1)
     {
      bcc = bp2304_calculate_bcc(swap_buf);
      if(r_bcc != bcc)
      {
       ComWrtByte(dev_port[instrID],NAK);
       if(bcc_repeat >= max_bcc_repeat)
    return (300);                /* NAK => MAX Repeat */
       bcc_repeat++;
      }
      else
      {
       ComWrtByte(dev_port[instrID],ACK);
       CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
       return(0);                /* 0 = ACK == OK*/
      }
     }
     else    /* Without bcc check*/
     {
      ComWrtByte(dev_port[instrID],ACK);
      CopyString (receive_buf, 0,in_buf,0,l-3); /* Copy without LF,ETX*/
      return(0);                /* 0 = ACK == OK*/
     }
    }

    return(0);
}

/*=========================================================================*/
/* This function send a comand and message                                 */
/*=========================================================================*/
int _VI_FUNC bp2304_send_x3_28_message(int instrID,char* cmd,char* msg)
{
    int ret,l;
    int bcc;

    ret = CompareStrings (msg, 0,"", 0, 0); /* Send a command without msg ?*/
    if(ret == 0)
     Fmt(out_buf,"%s<%c%x%x%x%x%s%c", EOT,
                      dev_group_addr[instrID],
                      dev_group_addr[instrID],
                      dev_user_addr[instrID],
                      dev_user_addr[instrID],
                      cmd,ENQ); /* without msg*/
    else
    {
     Fmt(swap_buf ,"%s<%s%c",msg,ETX);   /* with msg*/
     if(bcc_state == 1)
     {
      bcc = bp2304_calculate_bcc(swap_buf);
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s%c", EOT,
                       dev_group_addr[instrID],
                       dev_group_addr[instrID],
                       dev_user_addr[instrID],
                       dev_user_addr[instrID],
                       cmd,STX,swap_buf,bcc);
     }
     else
      Fmt(out_buf,"%s<%c%x%x%x%x%s%c%s", EOT,
                     dev_group_addr[instrID],
                     dev_group_addr[instrID],
                     dev_user_addr[instrID],
                     dev_user_addr[instrID],
                     cmd,STX,swap_buf);
    }

   /* FmtOut("out %s",out_buf);*/

    l = StringLength(out_buf);
    ret = ComWrt(dev_port[instrID], out_buf, l);     /* Write the Message */
    if (rs232err != 0)
     return (230);

    return (0);
}
/*=========================================================================*/
/* This function calculate the block check                                 */
/*=========================================================================*/
int _VI_FUNC bp2304_calculate_bcc(char* msg)
{
    int l,i,bcc;

    bcc = 0;
    l = StringLength(msg);
    for(i = 0;i <= l;i++)
     bcc ^= msg[i];     /* exclusive OR */

    bcc |= 0x80;    /* OR Bit 8 */

    return (bcc);
}

/* = Global UTILITY ROUTINES ==============================================*/
/*=========================================================================*/
/* Function: Invalid Integer Range                                         */
/* Purpose:  This function checks an integer to see if it lies between a   */
/*           minimum and maximum value.  If the value is out of range, set */
/*           the global error variable to the value err_code.  If the      */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC bp2304_invalid_integer_range (int val,int min,int max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp2304_err = err_code;
    return -1;
  }
  return 0;
}
/*=========================================================================*/
/* Function: Invalid Real Range                                            */
/* Purpose:  This function checks a real number to see if it lies between  */
/*           a minimum and maximum value.  If the value is out of range,   */
/*           set the global error variable to the value err_code.  If the  */
/*           value is OK, error = 0.                                       */
/*=========================================================================*/
int _VI_FUNC bp2304_invalid_real_range (double val,double min,double max,int err_code)
{
  if ((val < min) || (val > max))
  {
    bp2304_err = err_code;
    return -1;
  }
  return 0;
}
void bp2304_setup_arrays (void)
{
    int i;

    if(!setup_first)/*----------- Reset the Arrays before initial --------*/
    {
     for (i=1; i< bp2304_MAX_INSTR; i++)
     {
       dev_group_addr[i] = -1;
       dev_user_addr[i]  = -1;
       dev_port[i]       =  0;
     }
     setup_first = 1;
    }
    port_addr[0] = 0x3f8;
    port_addr[1] = 0x2f8;
    port_addr[2] = 0x3e8;
    port_addr[3] = 0x2e8;

    int_level[0] = 4;
    int_level[1] = 3;
    int_level[2] = 4;
    int_level[3] = 3;

    sense_frequenz[0] = ":SENSE:LFR 16.7";
    sense_frequenz[1] = ":SENSE:LFR 50";
    sense_frequenz[2] = ":SENSE:LFR 60";

    sense_range[0] = ":SENSE:RES:RANG:AUTO ON";
    sense_range[1] = ":SENSE:RES:RANG 100UOHM";
    sense_range[2] = ":SENSE:RES:RANG 1MOHM";
    sense_range[3] = ":SENSE:RES:RANG 10MOHM";
    sense_range[4] = ":SENSE:RES:RANG 100MOHM";
    sense_range[5] = ":SENSE:RES:RANG 1OHM";
    sense_range[6] = ":SENSE:RES:RANG 10OHM";
    sense_range[7] = ":SENSE:RES:RANG 100OHM";
    sense_range[8] = ":SENSE:RES:RANG 1KOHM";
    sense_range[9] = ":SENSE:RES:RANG 10KOHM";

    sense_resolution[0] = ":SENSE:RES:RES MIN";
    sense_resolution[1] = ":SENSE:RES:RES DEF";
    sense_resolution[2] = ":SENSE:RES:RES MAX";

    source_shape[0] = ":SOUR:FUNC SQU";
    source_shape[1] = ":SOUR:FUNC PULSE";
    source_shape[2] = ":SOUR:FUNC CONS";

    source_level[0] = ":SOUR:CURR MIN";
    source_level[1] = ":SOUR:CURR DEF";
    source_level[2] = ":SOUR:CURR MAX";

    sense_loading[0] = ":SENS:RES:LOAD REAL";
    sense_loading[1] = ":SENS:RES:LOAD COMP";

    input_ground[0] = ":INPUT:LOW GROUND";
    input_ground[1] = ":INPUT:LOW FLOAT";

    system_keyb[0] = ":SYST:KEYB:ACC FULL";  /* Acces Full*/
    system_keyb[1] = ":SYST:KEYB:ACC RANG";  /* Acces measuremt+Range*/
    system_keyb[2] = ":SYST:KEYB:ACC MEAS";  /* Acces measuremt*/

}
/*= THE END ===============================================================*/
/*void main(void)
{
  char str_id[100],unit[10];
  double value;
  int id,i,ret,c;

  i = 0;

  bp2304_init_gpib (9, 12, 0, str_id, &id);
  bp2304_init_serial(1,9600,3.0,0,0,0,str_id,0,&id);


  ret = 0;
  c = 'j';
  while(c == 'j')
  {
   ret = 0;
   bp2304_meas_start (id);
   while(ret == 0)
   {
    ret = bp2304_meas_read (id, &value, unit);

    if(ret != 0)
     FmtOut("\nERROR : %i",ret);
    else
    {
     FmtOut("\nValue %f %s %i",value,unit,i);
     i++;
    }
   }
   FmtOut("\n Neuer Start j/n");
   c = GetKey();
  }
  bp2304_meas_stop (id);
  bp2304_close (id);
}   */

